import Image from "next/image";
import Link from "next/link";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { 
  Building2, 
  Shield, 
  Clock, 
  Truck, 
  CheckCircle, 
  Phone, 
  Mail,
  MapPin,
  Factory,
  Wrench,
  Users
} from "lucide-react";

export default function Home() {
  const features = [
    {
      icon: <Factory className="h-8 w-8" />,
      title: "In-House Production",
      description: "Manufacturing facility based in Sharjah with quality control"
    },
    {
      icon: <Truck className="h-8 w-8" />,
      title: "UAE-Wide Delivery",
      description: "Supply and installation across all Emirates"
    },
    {
      icon: <Wrench className="h-8 w-8" />,
      title: "Custom Solutions",
      description: "Tailored porta cabin designs to meet your requirements"
    },
    {
      icon: <Clock className="h-8 w-8" />,
      title: "Quick Delivery",
      description: "Fast manufacturing and delivery within few days"
    }
  ];

  const products = [
    {
      id: 1,
      name: "Office Porta Cabin",
      description: "Office porta cabin with attached bathroom and pantry",
      image: "/office-cabin-placeholder.jpg",
      category: "Site Office",
      features: ["Bathroom Attached", "Pantry", "Air Conditioning", "Vinyl Flooring"]
    },
    {
      id: 2,
      name: "Security Cabin",
      description: "Security porta cabin for 1, 2 or 3 person occupancy",
      image: "/security-cabin-placeholder.jpg",
      category: "Security Solutions",
      features: ["Multiple Sizes", "Secure Design", "Weather Resistant", "Easy Installation"]
    },
    {
      id: 3,
      name: "Toilet Units",
      description: "Porta cabin toilet units from 1 unit to customized up to 20",
      image: "/toilet-units-placeholder.jpg",
      category: "Sanitation",
      features: ["1-20 Units", "Customizable", "Hygienic Design", "Water Connections"]
    }
  ];

  return (
    <div className="min-h-screen">
      {/* Hero Section */}
      <section className="relative h-screen flex items-center justify-center bg-gradient-to-r from-blue-900/80 to-gray-900/80">
        <div className="absolute inset-0 bg-black/40 z-10"></div>
        <div className="absolute inset-0">
          <div className="w-full h-full bg-gradient-to-br from-blue-800 via-gray-800 to-slate-800"></div>
        </div>
        
        <div className="relative z-20 text-center text-white px-4 max-w-4xl mx-auto">
          <h1 className="text-5xl md:text-7xl font-bold mb-6 leading-tight">
            PORTA CABIN
            <span className="block text-blue-400">SOLUTIONS!</span>
          </h1>
          <p className="text-xl md:text-2xl mb-8 text-gray-200 max-w-2xl mx-auto">
            Site Office, Security Cabin, Toilet Units, Customized Cabins
            <br />
            Safe Industry Solutions That Save Your Valuable Time and Money
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-blue-600 hover:bg-blue-700 text-lg px-8 py-6">
              <Phone className="mr-2 h-5 w-5" />
              Contact Us
            </Button>
            <Button 
              size="lg" 
              variant="outline" 
              className="bg-white/10 border-white/20 text-white hover:bg-white/20 text-lg px-8 py-6"
            >
              View Our Cabins
            </Button>
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
              Safe Industry Solutions in House Production
            </h2>
            <p className="text-lg text-gray-600 max-w-3xl mx-auto">
              SAAM CABINS have an extensive range of standard cabins that can be constructed to your specifications. 
              We offer porta cabin ablution units, site offices, security cabins, and portable toilets with quality fittings and furnishings.
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {features.map((feature, index) => (
              <div key={index} className="text-center group">
                <div className="bg-blue-100 w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4 group-hover:bg-blue-200 transition-colors">
                  <div className="text-blue-600">
                    {feature.icon}
                  </div>
                </div>
                <h3 className="text-xl font-semibold text-gray-900 mb-2">
                  {feature.title}
                </h3>
                <p className="text-gray-600">
                  {feature.description}
                </p>
              </div>
            ))}
          </div>
        </div>
      </section>

      {/* Featured Products */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <div className="text-center mb-12">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
              Our Porta Cabin Solutions
            </h2>
            <p className="text-lg text-gray-600">
              Porta Cabin Supply and Installation We Provide All Across U.A.E
            </p>
          </div>
          
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
            {products.map((product) => (
              <Card key={product.id} className="overflow-hidden hover:shadow-lg transition-shadow">
                <div className="relative h-64 bg-gray-300">
                  <div className="absolute inset-0 bg-gradient-to-br from-blue-800 to-gray-800"></div>
                  <Badge className="absolute top-4 left-4 bg-blue-600">
                    {product.category}
                  </Badge>
                </div>
                
                <CardHeader>
                  <div>
                    <CardTitle className="text-xl font-bold">{product.name}</CardTitle>
                    <CardDescription className="mt-1">
                      {product.description}
                    </CardDescription>
                  </div>
                </CardHeader>
                
                <CardContent>
                  <div className="flex flex-wrap gap-2 mb-4">
                    {product.features.map((feature, index) => (
                      <Badge key={index} variant="secondary" className="text-xs">
                        {feature}
                      </Badge>
                    ))}
                  </div>
                  
                  <Button className="w-full bg-blue-600 hover:bg-blue-700">
                    <Phone className="mr-2 h-4 w-4" />
                    Get Quote
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>
          
          <div className="text-center mt-12">
            <Button variant="outline" size="lg" asChild>
              <Link href="/work">
                View All Our Cabins
              </Link>
            </Button>
          </div>
        </div>
      </section>

      {/* Mission Section */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4 text-center">
          <div className="max-w-4xl mx-auto">
            <h2 className="text-3xl md:text-4xl font-bold mb-6 text-gray-900">
              Our Mission
            </h2>
            <p className="text-xl mb-8 text-gray-600">
              To Provide Innovative Solutions Matching to Latest Trends
            </p>
            <p className="text-lg mb-8 text-gray-700 leading-relaxed">
              We pride ourselves in our ability to design and create affordable, customized and environmentally friendly 
              portable buildings for professional requirements. Our cabins are available for UAE delivery and can be 
              built to order within a few days with insulation, fire security, doors and windows, vinyl flooring, 
              split system air conditioner, and bathroom partitions.
            </p>
          </div>
        </div>
      </section>

      {/* Call to Action */}
      <section className="py-16 bg-blue-800 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Want To Work With Us? Hit The Button.
          </h2>
          <p className="text-xl mb-8 max-w-2xl mx-auto">
            Get a quote on our porta cabin solutions. Call us at +971 58 201 2073 or fill out our quote form.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-white text-blue-800 hover:bg-gray-100 px-8">
              <Phone className="mr-2 h-5 w-5" />
              Call Now: +971 58 201 2073
            </Button>
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white/10 px-8" asChild>
              <Link href="/contact">
                <Mail className="mr-2 h-5 w-5" />
                Let's Work Together
              </Link>
            </Button>
          </div>
        </div>
      </section>
    </div>
  );
}
